<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Experiment;
use App\Models\Media;
use App\Traits\HasMedia;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class ExperimentController extends Controller
{
    use HasMedia;

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $experiments = Experiment::all();
        return view('backend.experiments.index', compact('experiments'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        return view('backend.experiments.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'subtitle' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'tags' => 'nullable|string',
            'url' => 'nullable|url'
        ]);

        try {
            DB::beginTransaction();

            $tagsArray = $request->input('tags') ? array_map('trim', explode(',', $request->input('tags'))) : [];

            $experiment = Experiment::create([
                'title' => $request->input('title'),
                'subtitle' => $request->input('subtitle'),
                'description' => $request->input('description'),
                'tags' => $tagsArray,
                'slug' => Str::slug($request->input('title')),
                'url' => $request->input('url'),
            ]);

            $this->handleMedia($experiment, $request, 'media_files');

            DB::commit();

            return redirect()->route('backend.experiments.index')->with('success', 'Experiment created successfully!');

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Error creating experiment: ' . $e->getMessage());
            return redirect()->back()->withInput()->with('error', 'Failed to create experiment. Please try again.');
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Experiment  $experiment
     * @return \Illuminate\View\View
     */
    public function show(Experiment $experiment)
    {
        $experiment->load('media');
        return view('backend.experiments.show', compact('experiment'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Experiment  $experiment
     * @return \Illuminate\View\View
     */
    public function edit(Experiment $experiment)
    {
        $experiment->load('media');
        return view('backend.experiments.edit', compact('experiment'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Experiment  $experiment
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, Experiment $experiment)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'subtitle' => 'nullable|string|max:255',
            'description' => 'nullable|string',
            'tags' => 'nullable|string',
            'url' => 'nullable|url',
            'media_files.*' => 'nullable|file|mimes:jpeg,png,jpg,gif,mp4,mov,ogg,webm|max:20480',
        ]);

        try {
            DB::beginTransaction();

            $tagsArray = $request->input('tags') ? array_map('trim', explode(',', $request->input('tags'))) : [];

            $experiment->title = $request->input('title');
            $experiment->subtitle = $request->input('subtitle');
            $experiment->description = $request->input('description');
            $experiment->tags = $tagsArray;
            $experiment->url = $request->input('url');

            if ($experiment->isDirty('title')) {
                $experiment->slug = Str::slug($request->input('title'));
            }
            $experiment->save();

            $this->handleMedia($experiment, $request, 'media_files');

            DB::commit();

            return redirect()->route('backend.experiments.index')->with('success', 'Experiment updated successfully!');

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Error updating experiment: ' . $e->getMessage());
            return redirect()->back()->withInput()->with('error', 'Failed to update experiment. Please try again.');
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Experiment  $experiment
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Experiment $experiment)
    {
        try {
            DB::beginTransaction();

            $experiment->media()->each(function ($media) {
                Storage::disk('public')->delete($media->file_path);
                $media->delete();
            });

            $experiment->delete();

            DB::commit();
            return redirect()->route('backend.experiments.index')->with('success', 'Experiment deleted successfully!');

        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Error deleting experiment: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to delete experiment. Please try again.');
        }
    }
}