<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Insight;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class InsightController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $insights = Insight::orderBy('id', 'desc')->get();
        return view('backend.insights.index', compact('insights'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        return view('backend.insights.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255|unique:insights,title',
            'summary' => 'required|string',
            'keyword' => 'required|string|max:255',
            'color' => 'nullable|string|max:7', // E.g., #RRGGBB
        ]);

        try {
            Insight::create($request->only('title', 'summary', 'keyword', 'color'));

            return redirect()->route('backend.insights.index')->with('success', 'Insight created successfully! ✨');
        } catch (Exception $e) {
            Log::error('Error creating insight: ' . $e->getMessage());
            return redirect()->back()->withInput()->with('error', 'Failed to create insight. Please try again.');
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Insight  $insight
     * @return \Illuminate\View\View
     */
    public function show(Insight $insight)
    {
        return view('backend.insights.show', compact('insight'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Insight  $insight
     * @return \Illuminate\View\View
     */
    public function edit(Insight $insight)
    {
        return view('backend.insights.edit', compact('insight'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Insight  $insight
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, Insight $insight)
    {
        $request->validate([
            'title' => 'required|string|max:255|unique:insights,title,' . $insight->id,
            'summary' => 'required|string',
            'keyword' => 'required|string|max:255',
            'color' => 'nullable|string|max:7',
        ]);

        try {
            $insight->update($request->only('title', 'summary', 'keyword', 'color'));

            return redirect()->route('backend.insights.index')->with('success', 'Insight updated successfully! 🚀');
        } catch (Exception $e) {
            Log::error('Error updating insight: ' . $e->getMessage());
            return redirect()->back()->withInput()->with('error', 'Failed to update insight. Please try again.');
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Insight  $insight
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Insight $insight)
    {
        try {
            $insight->delete();
            return redirect()->route('backend.insights.index')->with('success', 'Insight deleted successfully! 🗑️');
        } catch (Exception $e) {
            Log::error('Error deleting insight: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to delete insight. Please try again.');
        }
    }
}
