<?php

namespace App\Http\Controllers\Backend;

use Exception;
use App\Models\Piece;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Storage;

class PieceController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        // Mendapatkan semua 'pieces' dan mengurutkannya berdasarkan ID terbaru
        $pieces = Piece::orderBy('id', 'desc')->get();

        // Mengirimkan data ke view untuk ditampilkan dalam daftar
        return view('backend.pieces.index', compact('pieces'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        // Menampilkan form untuk membuat 'piece' baru
        return view('backend.pieces.create');
    }

    public function store(Request $request)
    {
        // Hapus dd($request); setelah selesai debugging
        // dd($request);

        $request->validate([
            'category' => 'required|string|max:255',
            'title' => 'required|string|max:255|unique:pieces,title',
            'description_json' => 'required|json',
            'model3d_file' => 'nullable|file|mimes:glb|max:102400', // max 100MB
        ]);

        try {
            DB::beginTransaction();

            $model3dUrl = null;

            if ($request->hasFile('model3d_file')) {
                $model3dPath = $request->file('model3d_file')->store('models', 'public');
                $model3dUrl = Storage::url($model3dPath);
            }

            // --- Perbaikan Kritis di sini ---
            $descriptionArray = $request->input('description_json');

            $piece = Piece::create([
                'category' => $request->input('category'),
                'title' => $request->input('title'),
                // Menggunakan array yang sudah didecode
                'description' => $descriptionArray,
                'model3d' => $model3dUrl,
            ]);
            // --- End Perbaikan ---

            DB::commit();

            return redirect()->route('backend.pieces.index')->with('success', 'Piece created successfully! 🖼️');
        } catch (Exception $e) {
            DB::rollBack();

            // Coba hapus file jika terjadi error setelah upload (jika $model3dPath tersedia)
            if (isset($model3dPath)) {
                Storage::disk('public')->delete($model3dPath);
            }

            Log::error('Error creating piece: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);

            return redirect()->back()->withInput()->with('error', 'Failed to create piece. Please check logs. ' . $e->getMessage());
        }
    }
    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Piece  $piece
     * @return \Illuminate\View\View
     */
    public function show(Piece $piece)
    {
        // Menampilkan detail dari 'piece' tertentu
        // Karena 'description' sudah di-cast ke array, Anda bisa mengaksesnya sebagai array di view
        return view('backend.pieces.show', compact('piece'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Piece  $piece
     * @return \Illuminate\View\View
     */
    public function edit(Piece $piece)
    {
        // Menampilkan form untuk mengedit 'piece'
        return view('backend.pieces.edit', compact('piece'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Piece  $piece
     * @return \Illuminate\Http\RedirectResponse
     */
    // Di dalam App\Http\Controllers\Backend\PieceController.php

    // ... (Use statements)

    // Di dalam App\Http\Controllers\Backend\PieceController.php

    // ... (Use statements)

    public function update(Request $request, Piece $piece)
    {
        // 1. Validasi
        $request->validate([
            'category' => 'required|string|max:255',
            // 'unique' diabaikan untuk ID yang sedang diedit
            'title' => 'required|string|max:255|unique:pieces,title,' . $piece->id,
            'description_json' => 'required|json', // Diubah dari nullable
            'model3d_file' => 'nullable|file|mimes:glb|max:102400', // File baru opsional
        ]);

        try {
            DB::beginTransaction();

            $model3dUrl = $piece->model3d; // Ambil URL lama secara default

            // 2. Proses File Upload BARU
            if ($request->hasFile('model3d_file')) {
                // Hapus file lama jika ada (hanya jika piece->model3d bukan null dan bukan URL eksternal)
                if ($piece->model3d) {
                    // Konversi URL kembali ke path storage jika menggunakan Storage::url()
                    $oldPath = Str::after($piece->model3d, '/storage/');
                    if (Storage::disk('public')->exists($oldPath)) {
                        Storage::disk('public')->delete($oldPath);
                    }
                }

                // Simpan file baru
                $model3dPath = $request->file('model3d_file')->store('models', 'public');
                $model3dUrl = Storage::url($model3dPath);
            }

            // 3. Update Data
            $piece->update([
                'category' => $request->input('category'),
                'title' => $request->input('title'),
                // Wajib DECODE karena dikirim sebagai JSON string dari JS
                'description' => json_decode($request->input('description_json'), true),
                'model3d' => $model3dUrl, // Gunakan URL baru/lama
            ]);

            DB::commit();

            return redirect()->route('backend.pieces.index')->with('success', 'Piece updated successfully! ✨');
        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Error updating piece: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            return redirect()->back()->withInput()->with('error', 'Failed to update piece. ' . $e->getMessage());
        }
    }
    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Piece  $piece
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Piece $piece)
    {
        try {
            DB::beginTransaction();

            // Hapus record piece
            $piece->delete();

            DB::commit();
            return redirect()->route('backend.pieces.index')->with('success', 'Piece deleted successfully! 🗑️');
        } catch (Exception $e) {
            DB::rollBack();
            Log::error('Error deleting piece: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to delete piece. Please try again.');
        }
    }
}
