<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Service;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class ServiceController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        // Mengambil semua layanan dan diurutkan berdasarkan ID atau kolom lain jika ada
        $services = Service::all();
        return view('backend.services.index', compact('services'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        return view('backend.services.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255|unique:services,title',
            'description' => 'required|string',
            'icon' => 'nullable|string|max:50',
        ]);

        try {
            Service::create($request->all());

            return redirect()->route('backend.services.index')->with('success', 'Service created successfully! ✨');
        } catch (Exception $e) {
            Log::error('Error creating service: ' . $e->getMessage());
            return redirect()->back()->withInput()->with('error', 'Failed to create service. Please try again.');
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Service  $service
     * @return \Illuminate\View\View
     */
    public function edit(Service $service)
    {
        return view('backend.services.edit', compact('service'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Service  $service
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, Service $service)
    {
        $request->validate([
            'title' => 'required|string|max:255|unique:services,title,' . $service->id,
            'description' => 'required|string',
            'icon' => 'nullable|string|max:50',
        ]);

        try {
            $service->update($request->all());

            return redirect()->route('backend.services.index')->with('success', 'Service updated successfully! 🚀');
        } catch (Exception $e) {
            Log::error('Error updating service: ' . $e->getMessage());
            return redirect()->back()->withInput()->with('error', 'Failed to update service. Please try again.');
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Service  $service
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Service $service)
    {
        try {
            $service->delete();
            return redirect()->route('backend.services.index')->with('success', 'Service deleted successfully! 🗑️');
        } catch (Exception $e) {
            Log::error('Error deleting service: ' . $e->getMessage());
            return redirect()->back()->with('error', 'Failed to delete service. Please try again.');
        }
    }
}
