<?php

namespace App\Services;

use App\Models\Visit;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;

class AnalyticsService
{
    /**
     * Metode utama untuk mencatat kunjungan dan memanggil API.
     */
    public function recordVisit(Request $request): void
    {
        $ipAddress = $request->ip();
        $url = $request->fullUrl();
        $sessionId = $request->session()->getId();

        // 1. Cek: Gunakan flash session untuk mencegah pencatatan ganda saat refresh cepat
        $hitKey = 'recorded_hit.' . md5($url);
        if ($request->session()->has($hitKey)) {
            return;
        }

        // 2. Panggil fungsi geolokasi (private)
        $geoData = $this->getGeolocation($ipAddress);

        // 3. Simpan data ke database
        Visit::create([
            'ip_address' => $ipAddress,
            'session_id' => $sessionId,
            'url' => $url,
            'referrer' => $request->header('referer'),
            'user_agent' => $request->header('User-Agent'),
            'country' => $geoData['country'] ?? null,
            'city' => $geoData['city'] ?? null,
        ]);

        // Tandai sebagai sudah dicatat
        $request->session()->flash($hitKey, true);
    }

    /**
     * Memanggil API IP-API untuk mendapatkan data geolokasi (Negara dan Kota).
     * Ini adalah logika eksternal yang terisolasi.
     */
    private function getGeolocation(string $ip): array
    {
        // Skip IP lokal/privat
        if ($ip === '127.0.0.1' || $ip === '::1' || filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE)) {
             return ['country' => 'Local', 'city' => 'Local'];
        }

        try {
            // IP-API: free, no auth, max 45 requests/minute (cukup untuk web pribadi)
            $response = Http::timeout(2)
                            ->get("http://ip-api.com/json/{$ip}?fields=country,city");
            
            if ($response->successful() && $response->json()) {
                return $response->json();
            }

        } catch (\Exception $e) {
            // Jika terjadi error (misalnya API down atau timeout), catat saja 'Unknown'
            // \Log::error("Geolocation API failed for IP: {$ip}", ['error' => $e->getMessage()]); 
        }
        
        return ['country' => 'Unknown', 'city' => 'Unknown'];
    }
}